/*
 * pushToWebDatabase.jsx
 * Javascript for InDesign CC, CS6, CS5.5, CS5
 * Version date: 20200225
 *
 * PURPOSE:
 * ========
 * Control BatchXSLT to push XML data which was exported 
 * with "BatchXSLT for InDesign" into the full-text search database
 *
 * The settings from the main menu are stored in the file 'zz_Settings/settingsFTP.set'.
 * These settings are reloaded every time this program is started.
 *
 *
 * IMPORTANT NOTES
 * ===============
 * 1. The database must exists and we must have SELECT, INSERT, UPDATE rights
 *
 * DISCLAIMER:
 * ===============
 * Absolutely no warranty. Use it as is or modify it to match your needs
 *
 * Author: Andreas Imhof, www.aiedv.ch
 *
 * History:
 *    20200225 - Ai: Changed to control the freeware version
 *		20200123 - Ai: Runs with InDesign 2020 on Catalina too
 *		20171025 - Ai: Fixed a problem with font conversion tables
 * 		20170223 - Ai: General package new version 23.06.44
 * 		20140802 - Ai: Support CC 2014 v 10.x.x.x
 * 		20140708 - Ai: added expandable hooks at certain points
 *						afterInit, beforeDBpush, afterDBpush,
 *						beforeInitMainDialog, beforeShowMainDialog, afterShowMainDialog
 *
 */

// ****** Configurable SETTINGS ******
var copyright = "\u00a9www.AiEDV.ch";
var version = "40.0 46 - CC-CS5";   // Transformer and Scripts version
var packageExtension = "";        // set to an extension like '_PE' to a dedicated package installation or empty for the standard package installation
                                    // this expands the main application installation path to
                                    // BXSLT4InDesignBasePath = "~/applicationProgramName" + packageExtension + "/" + tranformerName + "/"
var tranformerName = "BatchXSLT";
var applicationHumanName = tranformerName + " for InDesign";
var applicationProgramName = tranformerName + "4InDesign";
var BXSLT4InDesignAppl = tranformerName + ".app";
var folderSeparator = "/";	// default to OSX
var os = "";
var programFolder = "";
getOS();

var	appVersion = app.version;
var majorINDVersion = 7;	// assume CS5

var settingsFileName = "settingsDB.set";  // name of the settings file to load. default = "settingsDB.set"
var settingsFilePath = "~/" + applicationProgramName + "Comm" + packageExtension + "/Scripts/zz_Settings";	// we must store settings somewhere into user home to have write access
var settingsFilePathAlt = "zz_Settings";							// if we have a problem try InDesign Application's script folder.

                          // The default path for XML data exported from InDesign
                          // (a path starting with tilde '~' points to the users home directory)
                          // -- set it to any existing path which should act as default export folder
                          //    and no folder chooser dialog will be shown
                          // -- set to "" to show a dialog to choose the export folder
var mainExportFolder = "";

var sourcePathName = "";
var xslPathName = "Utilities/DataBase/BIND2MySQL_ft_flipbook.xsl";
var outputPathName = "*NONE*";

var db_url = "";
var db_port = "3306";
var db_name = "";
var db_tablename = "";
var db_user = "";
var db_pw = "";

var domain = "";
var root_datapath = "";
var move2WEBdone = false;
var PDFtextExtract = false;


var saveSettings = true;                // true to save settings when starting export
var silent = false;                     // true to show NO completion message(s), false otherwise

                                        // The path to the folder where InDesign and transformer will communicate 
var BXSLT4InDesignCommDirDefault = "~/" +  applicationProgramName + "Comm" + packageExtension + "/" + tranformerName + "/";
var BXSLT4InDesignCommDir = BXSLT4InDesignCommDirDefault;

var BXSLT4InDesignBasePathDefault = programFolder + "/" + applicationProgramName + packageExtension + folderSeparator + tranformerName + folderSeparator;	// without the root/ProgramFilesFolder
var BXSLT4InDesignBasePathDefault2 = "~/" + applicationProgramName + packageExtension + folderSeparator + tranformerName + folderSeparator;
loadCommPrefsFromFile();
var BXSLT4InDesignBasePath = BXSLT4InDesignBasePathDefault;
var BXSLT4InDesignPath = BXSLT4InDesignBasePath + BXSLT4InDesignAppl;
                                        // The path to the folder where InDesign and tranformer will communicate 

// ****** END OF Configurable SETTINGS ******
var windowTitle = applicationHumanName + " - Full-Text Database Transfer  v" + version + "    '" +  settingsFileName + "'    " + copyright;

// ERROR codes
var last_errorCode = 0,
	error = {
		text : "",
		lastcode : 0,
		NOERR : 0,
		WRONG_INDD_VERSION : -1,
		EXPORTPATH_NOT_FOUND : -2,
		EXPORT_CANCELLED : -12
	};

var silentMode = 0;                     // Flags to combine for desired Messages
                                        // -1 = absolutely no message and dialog
                                        // 0 = show main export dialog and processing and completion messages
                                        // 1 = no main dialog

var mainDialogErrorMessage = "",
	mainDialogErrorCode = 0;


var BXSLTstarted = false;

var scriptsPath = getScriptsPath(),
	scriptName = getScriptName();

var useHooks = true,			// do check for hooks to call
	hooksBasePath = scriptsPath + folderSeparator + "zz_Plugins" + folderSeparator + "hooksDB" + folderSeparator,			// the base path to all hooks folders
	hooksPath = "",				// the path to certain hooks folders
	hooksFolder = null,			// the hooks folder object
	hooksScripts = null,		// the hooks folder scripts files list
	hooks = {
		/* object of this type:
		"settingsFileName": {		// hook for certain settings
								"callerscriptname":scriptName,	// the caller script name
								"hook":"hook name",				// the hook name
								"args":null,	// arguments passed to scripts called by doScript (the doScripts arguments param is not working)
												// can be of any type, the callee must know how to handle it. usually an object.
								"retval":0		// the return value set by a script called by doScript (the doScripts does not return anything)
							}
		*/
	},

	dbPushReRun = true,		// set true to enter the run loop
	dbPushRun = 0;


// determine a writable settings folder (in application or in home)
settingsFilePath = settingsFilePathWritable();
// load previously saved settings
loadSettingsFromFile();




// initialize and start export
app.scriptPreferences.userInteractionLevel = UserInteractionLevels.interactWithAll;

// allow hook to modify initialization
runHooks(scriptName,"afterInit", null);


/****************************
 * init some stuff.....let's go
 */
while (dbPushReRun == true) {
	var goahead = false;

	dbPushReRun = false;
	dbPushRun++;
	// run hooks before DB push
	runHooks(scriptName,"beforeDBpush", null);

	goahead = init();

	// run hooks after DB push
	if (goahead == 1) runHooks(scriptName,"afterDBpush", null);	// run hooks if not cancelled by user
}

exit();

/* ==================== FUNCTIONS ==================== */

/****************************
 * get operating system
 */
function getOS() {
	os = $.os;
	do {
		if (os.toLowerCase().indexOf("macintosh") >= 0) { 
			os = "mac";
			folderSeparator = "/";
			programFolder = folderSeparator + "Applications";
			BXSLT4InDesignAppl = tranformerName + ".app";
			break;
		}
		if (os.toLowerCase().indexOf("wind") >= 0) {
			os = "win";
			folderSeparator = "\\";
			programFolder = $.getenv("ProgramFiles");
			BXSLT4InDesignAppl = tranformerName + ".bat";
			break;
		}
		os = "unix";
		folderSeparator = "/";
		programFolder = folderSeparator + "Applications";
		BXSLT4InDesignAppl = tranformerName + ".sh";
	} while(false);
}


/****************************
 * hooks functions
 */
function getHookretval(settingsfilename) {
	if (!hooks[settingsFileName]) return 0;
	return 0;
}
function runHooks(callerscriptname, whichhook, args) {
	var hi;
	if (callerscriptname == "") {
		alert("Hooks callerscriptname is not set!! Hook call aborted!");
		return -1;
	}
	if (useHooks) {
		// export is DONE: check for hook to run after export
		hooksPath = hooksBasePath + whichhook;
		hooksFolder = new Folder(hooksPath);
		if (hooksFolder.exists) {
			hooksScripts = hooksFolder.getFiles("*.jsx*");	// get .jsx and .jsxbin
			for (hi = 0; hi < hooksScripts.length; hi++) {
				//alert("caller: " + callerscriptname + "\nhook " + whichhook + "\nscript: " + hooksScripts[hi].fsName);
				if (startsWith(hooksScripts[hi].name,"__") == false) {	// skip those starting with two underscores
					hooks[settingsFileName] = {"callerscriptname":callerscriptname,"hook":whichhook,"args":args,"retval":0};
					app.doScript(hooksScripts[hi],ScriptLanguage.javascript);
				}
			}
		}
	}
	return 0;
}


/****************************
 * init some stuff.....
 */
function init() {
	var goahead = 1;	// 0 = cancelled dialog
							// 1 = do FTP push
							// 2 = load new settings
							// 3 = set factory settings

	// run with CS5 7.x.x, CS6 8.x.x, CC 9.x.x, CC 2014 10.x.x
	var versParts = appVersion.split(".");
	majorINDVersion = parseInt(versParts[0],10);
	if (majorINDVersion < 7) {
		alert("This export script may not run with this version of InDesign!\nIt is designed to work with InDesign CS5 !\n\nProcessing aborted.");
		last_errorCode = error.WRONG_INDD_VERSION;
		return(error.WRONG_INDD_VERSION);
	}


	// check if tranformer application may be found
	var bxsltappl_found = findBXSLTappl();
	if (bxsltappl_found == false) return -1;

	build_export_paths();

	// run hooks before DB push
	runHooks(scriptName,"beforeInitMainDialog", null);

	if (getHookretval(settingsFileName) != -1) {	// show main dialog if not -1 is returned
		// show main dialog
		if (!(silentMode & 1)) {
			while (true) {
				goahead = showMainDialog();
				switch (goahead) {
					case 0: return(error.EXPORT_CANCELLED);	// cancelled
					case 2:	// load settings file
						break;
					case 3:	// save settings
						break;
					case 4:	// set factory settings
						break;
					default: break;	// go, do export
				}
				//alert("goahead: " + goahead);
				if ((goahead == 2) || (goahead == 3) || (goahead == 4)) continue;	// revert or load default settings
				var fieldsCheck = checkDialogFields();
				if (fieldsCheck != 0) {
					continue;
				}
				break;
			}
		}
	}

	// check if Export folder structure may be found
	if (sourcePathName != "") {
		var f = new File(sourcePathName);
		if (f.exists == false) {
			alert("The source data folder could not be found at:\n" + sourcePathName + "\n\nPlease, verify your data.");
			return -2;
		}
	}

	// write override jobticket
	var jobticket_written = newJobTicketBatchXSLT(BXSLT4InDesignCommDir);

	// fire tranformer to push XML into full-text database
	if (jobticket_written) {
		var called = callBXSLT_push();
	}

	return goahead;
}


/****************************
 * start transformer
 */
function callBXSLT_push() {

	// start Transformer
	BXSLTstarted = false;
	if (BXSLT4InDesignPath != "") {
		var app2start = new File (BXSLT4InDesignPath);
		BXSLTstarted = app2start.execute();
		if (BXSLTstarted == false) {
			alert(applicationHumanName + " could not be started!\nStart it manually!!");
			return(false);
		}
		else {
		}
	}

	return(true);
}




function loadCommPrefsFromFile() {
	// load the settings to communicate with tranformer from disk
	var commPrefsFile = new File(BXSLT4InDesignCommDir + "/comm/comm.prefs");
	if (commPrefsFile.exists == false) return;
	var err = commPrefsFile.open("r");

	var line = "";
	while (!commPrefsFile.eof) {
		line = commPrefsFile.readln();
		if ((line != null) && (line != "")) {
			var args_arr = line.split("=");
			switch(args_arr[0]) {
				case "BatchXSLTMajorVersion": BatchXSLTMajorVersion = args_arr[1]; break;
				case "BatchXSLTMinorVersion": BatchXSLTMinorVersion = args_arr[1]; break;
				case "BatchXSLTRequiredXslVersionName": BatchXSLTRequiredXslVersionName = args_arr[1]; break;
				case "BatchXSLTAppDir": BXSLT4InDesignBasePathDefault = args_arr[1]; break;
				case "BatchXSLTCommDir": BXSLT4InDesignCommDir = args_arr[1]; break;
			}
		}
	}
	err = commPrefsFile.close();
}


/****************************
 * check if tranformer application may be found
 */
function findBXSLTappl() {
	var found = false,
		runcnt = 0;
	do {
		runcnt++;
		switch (runcnt) {
			case 1:	// try the path currently set by loadCommPrefsFromFile
				// BXSLT4InDesignBasePath already IS set
				break;
			case 2:	// try default path in user.home - may be a downloaded version for testing only
				BXSLT4InDesignBasePath = BXSLT4InDesignBasePathDefault;
				break;
			case 3:	// try default path in /Applications (OSX) or %Programfiles% (WIN)
				BXSLT4InDesignBasePath = BXSLT4InDesignBasePathDefault2;
				break;
			case 4:	// try default path in %Programfiles% (WIN) without ' (x86)'
				if (os == 'win') {
					var bp = BXSLT4InDesignBasePathDefault2
					if (bp.indexOf(' (x86)') >= 0) bp = bp.replace(' (x86)','');
					BXSLT4InDesignBasePath = bp;
				} else continue;
				break;
			case 5:	// try to find any folder in /Applications containing 'applicationProgramName*'
				var fldr = new Folder(programFolder);
				var applFldrs = fldr.getFiles(applicationProgramName + packageExtension + "*");
				if (applFldrs.length < 1) break;	// let user choose the application
				for (var i = 0; i < applFldrs.length; i++) {
					var afldr = applFldrs[i];
					var applfldr = afldr + folderSeparator + tranformerName + folderSeparator + tranformerName + ".app";
					//alert(applfldr);
					fldr = new File(applfldr);
					if (fldr.exists == true) {
						BXSLT4InDesignBasePath = afldr + folderSeparator + tranformerName + folderSeparator;
						//alert("found appl: " + applfldr);
						break;
					}
				}
				break;	// let user choose the application
			default:	// this will open the file chooser
		}
		// set transformer application to call
		BXSLT4InDesignPath = BXSLT4InDesignBasePath + BXSLT4InDesignAppl;
		//alert("runcnt:"+runcnt +"\n"+ BXSLT4InDesignPath);
		var f = new File(BXSLT4InDesignPath);
		if (f.exists == false) {
			if (runcnt < 5) continue;
			alert("The '" + applicationHumanName + "' application can not be found at\n" + BXSLT4InDesignPath + "\n\nPlease, copy the application to your HOME folder (as mentioned in the manual) or choose the application in the following dialog and try again!");
			var bxslt_appl_fldr = Folder.selectDialog("Please select the folder containing the tranformer application");	// "Please select the folder containing the tranformer application"
			if (bxslt_appl_fldr == null) {
				alert("Database Push aborted.\n\nThe application may not continue because the Transformer application may not be found.");
				return(false);
			}
			if (bxslt_appl_fldr != null) BXSLT4InDesignBasePath = bxslt_appl_fldr + folderSeparator;
			continue;
		}
		found = true;
		break;
	} while(true);
	return(found);
}




/****************************
 * create a jobticket for tranformation
 */
function newJobTicketBatchXSLT(path) {
	var jtname = new Date().getTime() + "override.jt",
		jtFile = null,
		commpath = path,
		f, fldr,
		err;

	if (path == null || path == "") return;
	f = new File(path);
	if (f.exists == false) {
		fldr = new Folder(path);
		fldr.create();
		f = new File(path);
		if (f.exists == false) {
			alert("## An error ocurred while creating the communication folder\n" + BXSLT4InDesignCommDir + "\n\nTransfer failed");
			return;
		}
	}

	var sourceFileActionPath = "";
	if (move2WEBdone == true) {
		sourceFileActionPath = mainExportFolder;
		if (endsWith(sourceFileActionPath,"/") == false) sourceFileActionPath += "/";
		sourceFileActionPath += "WEBdone/";
	}

	//-----------------------------------------
	// write the jobticket to fill the full-text db
	my_sourcePathName = sourcePathName.replace(/\\/gi,"/");

	if (endsWith(commpath,folderSeparator) == false) commpath += folderSeparator;
	jtFile = new File(commpath + jtname);
	jtFile.encoding = "UTF-8";
	err = jtFile.open("w");

	err = jtFile.write("init_previous_jobticket = 1\n");
	err = jtFile.write("jobticketVersion = 10.0\n");
	err = jtFile.write("applicationPurpose = InDesign_INX_Transformer\n");
	err = jtFile.write("mode = 1\n");
	err = jtFile.write("logfileWrite = 1\n");
	err = jtFile.write("\n");
	err = jtFile.write("jt_triggerfile = \n");
	err = jtFile.write("sourcePathName = " + my_sourcePathName + "\n");
	err = jtFile.write("xslPathName = " + xslPathName + "\n");
	err = jtFile.write("outputPathName = " + outputPathName + "\n");
	err = jtFile.write("sourceFileAction = " + sourceFileActionPath + "\n");
	err = jtFile.write("excludeCleanupRunFileNameExts = XSLCSS\n");
	err = jtFile.write("excludeSourceFileNameExts = XSLCSS,_int.xml,.xmi\n");
	err = jtFile.write("deleteSourceDirs = 0\n");
	err = jtFile.write("nextJobTicketPath =\n");
	err = jtFile.write("nextJobTicketFileName = autostart.jt\n");

	err = jtFile.write("tp_1 = db_type=jdbc:mysql\n");
	err = jtFile.write("tp_2 = db_drvr=com.mysql.jdbc.Driver\n");
	err = jtFile.write("tp_3 = db_url=" + db_url + "\n");
	err = jtFile.write("tp_4 = port=" + db_port + "\n");
	err = jtFile.write("tp_5 = db=" + db_name + "\n");
	err = jtFile.write("tp_6 = dbtbl=" + db_tablename + "\n");
	err = jtFile.write("tp_7 = user=" + db_user + "\n");
	err = jtFile.write("tp_8 = pw=" + db_pw + "\n");
	err = jtFile.write("tp_9 = DOMAIN=" + domain + "\n");
	err = jtFile.write("tp_10 = root_datapath=" + root_datapath + "\n");
	err = jtFile.write("tp_11 = sub_path_override=\n");
	err = jtFile.write("tp_12 = sub_path=&&INPUT_SUBPATH&\n");
	err = jtFile.write("tp_13 = src_name=&&SRC_FILENAME&\n");
	err = jtFile.write("tp_14 = encoding=UTF8\n");
	err = jtFile.write("tp_15 = DEBUG=1\n");
	if (PDFtextExtract == true) err = jtFile.write("tp_16 = PDFtextExtract=1\n");
	else err = jtFile.write("tp_16 = PDFtextExtract=0\n");

	err = jtFile.close();

	writeOverrideQueue(commpath,jtname);

	return(err);
}


function writeOverrideQueue(path,jtname,jtname1) {
	if ((path == null) || (path == "")) return(-1);
	var commpath = path;
	if (endsWith(commpath,folderSeparator) == false) commpath += folderSeparator;
	var queuename = "override.que";
	var queuepath = commpath + queuename;
	var xqueuename = "x" + queuename;
	var xqueuepath = commpath + xqueuename;
	jtQfile = new File(queuepath);
	var err = false;
	if (jtQfile.exists == true) {	// try to rename queue file if exists
		for (var i = 0; i < 1000; i++) {	// try to rename file
			err = jtQfile.rename(xqueuename);
			if (jtQfile.error != "") ;	//alert("rename fserror: " +jtQfile.error);
			if (err == true) break;
			//alert("locked");
			$.sleep(10);
		}
		if (err == false) {
			//alert("error rename to: " + xqueuename);
			return(-1);	// uuups
		}
	}

	jtQfile = new File(xqueuepath);
	jtQfile.encoding = "UTF-8";
	for (var i = 0; i < 100; i++) {	// try to open until file is free. don't wait too long
		err = jtQfile.open("e");
		if (err == true) break;
		$.sleep(100);
	}
	if (err == false) return(-1);
	if (jtQfile.error != "") alert("fserror: " +jtQfile.error);

	err = jtQfile.seek(0,2);	// seek to eof to append
	//if (err != true) alert("error seek");
	if ((jtname != null) && (jtname != "")) err = jtQfile.writeln(jtname);
	//if (err != true) alert("error jt write");
	if ((jtname1 != null) && (jtname1 != "")) err = jtQfile.writeln(jtname1);
	//if (err != true) alert("error jt1 write");
//	alert("wait to close file");
	err = jtQfile.close();
	//if (err != true) alert("error close");
	err = jtQfile.rename(queuename);
	//if (err != true) alert("error rename back to: " + queuename);
	if (err == false) return(-1);
	return(0);
}




/****************************
 * SHOW THE STARTING MAIN DIALOG
 */
function showMainDialog() {
	var outfldr = new Folder(sourcePathName),
		trans = new Folder(BXSLT4InDesignBasePath),
		dlgw,

		windowFont,
		font,

		leftGroupWidth,
		leftGroupSiz,
		spacerSize1,
		labelSize,
		editFieldSize,
		editFieldSize2;

	windowTitle = applicationHumanName + " - Full-Text Database Transfer  v" + version + "    '" +  settingsFileName + "'    " + copyright;
	dlgw = new Window('dialog', windowTitle, undefined, {resizeable:false, closeButton: true, maximizeButton:false, minimizeButton:false});
	dlgw.orientation = 'row';
	dlgw.margins = [7,7,7,7];
	dlgw.alignChildren = ['left','top'];
	dlgw.alignment = ['left','top'];

	windowFont = dlgw.graphics.font;
	font = ScriptUI.newFont ( windowFont.name, windowFont.name.style, 11 );
	leftGroupWidth = 600;
	leftGroupSize = [leftGroupWidth,300];
	spacerSize1 = [10,15];
	labelSize = [130,2.0*font.size];
	editFieldSize = [200,2.0*font.size];
	editFieldSize2 = [400,2.0*font.size];


	with(dlgw){
		//------------------------------
		// items on left side
		leftGroup = add('group');
		leftGroup.orientation = 'column';
		leftGroup.alignment = ['left','top'];
		leftGroup.alignChildren = ['left','top'];
		leftGroup.spacing = 5;

		with(leftGroup) {
			if (mainDialogErrorMessage != "") {	// show the error message
				var errmessage = "Error #" + mainDialogErrorCode + " in Dialog fields: " + mainDialogErrorMessage;
				errormess = add('statictext', undefined, errmessage);
			}

			sourcePathGroup = add('group', undefined);
			sourcePathGroup.orientation = 'row';
			sourcePathGroup.alignChildren = ['left','top'];
			sourcePathGroup.alignment = ['left','top'];
				sourcePathGroup.label = sourcePathGroup.add('statictext', undefined, "Source Data Path:");
					sourcePathGroup.label.graphics.font = font;
					sourcePathGroup.label.size = labelSize;
					sourcePathGroup.label.justify = "right";
				sourcePathGroup.sourcePathNameField = sourcePathGroup.add('edittext', undefined, sourcePathName);
					sourcePathGroup.sourcePathNameField.size = editFieldSize2;
					sourcePathGroup.sourcePathNameField.graphics.font = font;
					sourcePathGroup.sourcePathNameField.helpTip = "The path to the source data to push into database";

			// ------------------------- a spacer
			add('statictext', undefined, "").size = spacerSize1;

			// items on left side in left groug
			ll = add('group').add('panel', undefined, '  Database Connection  ');
			ll.orientation = 'column';
			ll.alignment = ['left','top'];
			ll.alignChildren = ['left','top'];
			ll.spacing = 5;

			// ad items in left left group
			with(ll) {
				dbURLGroup = add('group', undefined);
				dbURLGroup.alignment = ['left','top'];
				dbURLGroup.orientation = 'row';
				dbURLGroup.alignChildren = ['right','top'];
					dbURLGroup.urlLabel = dbURLGroup.add('statictext', undefined, "Database            URL:");
						dbURLGroup.urlLabel.graphics.font = font;
						dbURLGroup.urlLabel.size = labelSize;
						dbURLGroup.urlLabel.justify = "right";
					dbURLGroup.db_urlField = dbURLGroup.add('edittext', undefined, db_url);
						dbURLGroup.db_urlField.size = editFieldSize;
						dbURLGroup.db_urlField.graphics.font = font;
						dbURLGroup.db_urlField.helpTip = "The URL to the database";

					dbURLGroup.portLabel = dbURLGroup.add('statictext', undefined, "Port:");
						dbURLGroup.portLabel.graphics.font = font;
						dbURLGroup.portLabel.size = labelSize;
						dbURLGroup.portLabel.justify = "right";
					dbURLGroup.db_portField = dbURLGroup.add('edittext', undefined, db_port);
						dbURLGroup.db_portField.size = editFieldSize;
						dbURLGroup.db_portField.graphics.font = font;
						dbURLGroup.db_portField.helpTip = "The port on which the database answers";


				dbNameGroup = add('group', undefined);
				dbNameGroup.alignment = ['left','top'];
				dbNameGroup.orientation = 'row';
				dbNameGroup.alignChildren = ['right','top'];
					dbNameGroup.dbnameLabel = dbNameGroup.add('statictext', undefined, "Name:");
						dbNameGroup.dbnameLabel.graphics.font = font;
						dbNameGroup.dbnameLabel.size = labelSize;
						dbNameGroup.dbnameLabel.justify = "right";
					dbNameGroup.db_nameField = dbNameGroup.add('edittext', undefined, db_name);
						dbNameGroup.db_nameField.size = editFieldSize;
						dbNameGroup.db_nameField.graphics.font = font;
						dbNameGroup.db_nameField.helpTip = "The database name";

					dbNameGroup.usernameLabel = dbNameGroup.add('statictext', undefined, "User Name:");
						dbNameGroup.usernameLabel.graphics.font = font;
						dbNameGroup.usernameLabel.size = labelSize;
						dbNameGroup.usernameLabel.justify = "right";
					dbNameGroup.db_userField = dbNameGroup.add('edittext', undefined, db_user);
						dbNameGroup.db_userField.size = editFieldSize;
						dbNameGroup.db_userField.graphics.font = font;
						dbNameGroup.db_userField.helpTip = "The user name to use to login to the database";


				dbTableGroup = add('group', undefined);
				dbTableGroup.alignment = ['left','top'];
				dbTableGroup.orientation = 'row';
				dbTableGroup.alignChildren = ['right','top'];
					dbTableGroup.dbtableLabel = dbTableGroup.add('statictext', undefined, "Table Name:");
						dbTableGroup.dbtableLabel.graphics.font = font;
						dbTableGroup.dbtableLabel.size = labelSize;
						dbTableGroup.dbtableLabel.justify = "right";
					dbTableGroup.db_tablenameField = dbTableGroup.add('edittext', undefined, db_tablename);
						dbTableGroup.db_tablenameField.size = editFieldSize;
						dbTableGroup.db_tablenameField.graphics.font = font;
						dbTableGroup.db_tablenameField.helpTip = "The table name";

					dbTableGroup.passwordLabel = dbTableGroup.add('statictext', undefined, "Password:");
						dbTableGroup.passwordLabel.graphics.font = font;
						dbTableGroup.passwordLabel.size = labelSize;
						dbTableGroup.passwordLabel.justify = "right";
					dbTableGroup.db_pwField = dbTableGroup.add('edittext', undefined, db_pw);
						dbTableGroup.db_pwField.size = editFieldSize;
						dbTableGroup.db_pwField.graphics.font = font;
						dbTableGroup.db_pwField.helpTip = "The user's password";

				// ------------------------- a spacer
				add('statictext', undefined, "").size = spacerSize1;


				domainGroup = add('group', undefined);
				domainGroup.alignment = ['left','top'];
				domainGroup.orientation = 'row';
				domainGroup.alignChildren = ['right','top'];
					domainGroup.domainLabel = domainGroup.add('statictext', undefined, "Domain Name:");
						domainGroup.domainLabel.graphics.font = font;
						domainGroup.domainLabel.size = labelSize;
						domainGroup.domainLabel.justify = "right";
					domainGroup.domainField = domainGroup.add('edittext', undefined, domain);
						domainGroup.domainField.size = editFieldSize;
						domainGroup.domainField.graphics.font = font;
						domainGroup.domainField.helpTip = "The url to set in the database field 'domain'";

					domainGroup.passwordLabel = domainGroup.add('statictext', undefined, "Data Root Path:");
						domainGroup.passwordLabel.graphics.font = font;
						domainGroup.passwordLabel.size = labelSize;
						domainGroup.passwordLabel.justify = "right";
					domainGroup.root_datapathField = domainGroup.add('edittext', undefined, root_datapath);
						domainGroup.root_datapathField.size = editFieldSize;
						domainGroup.root_datapathField.graphics.font = font;
						domainGroup.root_datapathField.helpTip = "The path to set in the database field 'root_datapath'";
			}

			// ------------------------- a spacer
			add('statictext', undefined, "").size = spacerSize1;

			// the fields below ll		
			PDFtextExtractCheck = add("checkbox", undefined, "Extract Text from PDF files");
				PDFtextExtractCheck.value = PDFtextExtract;
				PDFtextExtractCheck.graphics.font = font;
				PDFtextExtractCheck.helpTip = "Check to extract text from PDFs";	// Page Preview JPEG

			saveSettingsCheck = add("checkbox", undefined, "Save these settings");
				saveSettingsCheck.value = saveSettings;
				saveSettingsCheck.graphics.font = font;
				saveSettingsCheck.helpTip = "Save these settings to a file on disk";

			move2WEBdoneCheck = add("checkbox", undefined, "Move to 'WEBdone' after complete");
				move2WEBdoneCheck.value = move2WEBdone;
				move2WEBdoneCheck.graphics.font = font;
				move2WEBdoneCheck.helpTip = "Move all processed files to 'WEBdone' folder after complete";


			// ------------------------- a spacer
			add('statictext', undefined, "").size = spacerSize1;

			add('statictext', undefined, "Transformer Path: " + trans.fsName);
		}




		//------------------------------
		// The buttons on right side
		rightGroup = dlgw.add('group');
		rightGroup.orientation = 'column';
		rightGroup.alignChildren = ['fill','top'];
		rightGroup.alignment = ['fill','top'];
			rightGroup.btnPanel = rightGroup.add('panel', undefined, '');
				rightGroup.btnPanel.goBtn = rightGroup.btnPanel.add('button', undefined, "GO", {name:'go'});
				rightGroup.btnPanel.goBtn.helpTip = "Start loading full-text database using the current settings";
				rightGroup.btnPanel.cancelBtn = rightGroup.btnPanel.add('button', undefined, "Cancel", {name:'cancel'});
				rightGroup.btnPanel.cancelBtn.helpTip = "Cancel database load";

			rightGroup.sub2Group = rightGroup.add('group');
			rightGroup.sub2Group.orientation = 'column';
			rightGroup.sub2Group.alignChildren = ['fill','top'];
			rightGroup.sub2Group.alignment = ['fill','top'];
			rightGroup.sub2Group.margins = [0,38,0,0];
				rightGroup.sub2Group.loadSettingsFromFileBtn = rightGroup.sub2Group.add('button', undefined, "Load Settings");
				rightGroup.sub2Group.loadSettingsFromFileBtn.helpTip = "Load new settings from settings file";	// Load new settings from settings file
				rightGroup.sub2Group.saveSettingsToFileBtn = rightGroup.sub2Group.add('button', undefined, "Save Settings");
				rightGroup.sub2Group.saveSettingsToFileBtn.helpTip = "Save settings to file";	// Save settings to file

		/*
			The dialog item handlers
		*/
		// Go and Cancel buttons
		rightGroup.btnPanel.goBtn.onClick = function() { dlgw.close(1); } 
		rightGroup.btnPanel.cancelBtn.onClick = function() { dlgw.close(0); } 
		// Settings buttons
		rightGroup.sub2Group.loadSettingsFromFileBtn.onClick = function() {
			var f = new Folder(settingsFilePath);
			var selected = f.openDlg("Choose Settings File");
			if (selected != null) {
				var fnew = new Folder(selected);
				settingsFilePath = fnew.path;
				settingsFileName = fnew.name;
				//alert("selected: " + selected + "\nfullpath: " + fnew.fsName + "\npath: " + fnew.path + "\nname: " + fnew.name);
				var loaded = loadSettingsFromFile();
				dlgw.close(2);
			}
		} 
		rightGroup.sub2Group.saveSettingsToFileBtn.onClick = function() {
			var savepath = settingsFilePath;
			if (endsWith(savepath,folderSeparator) == false) savepath += folderSeparator;
			savepath += settingsFileName;
			var f = new File(savepath);
			var selected = f.saveDlg("Name the Settings File");
			if (selected != null) {
				settingsFilePath = selected.path;
				settingsFileName = selected.name;
				settingsloaded = 0;
				if (endsWith(settingsFileName.toLowerCase(),".set") == false) settingsFileName += ".set";
				//alert("savepath: " + savepath + "\nselected: " + selected + "\nsettingsFilePath: " + settingsFilePath + "\nsettingsFileName: " + settingsFileName);
				dlgw.close(3);
			}
		} 
	}


	// before show main dialog: run hook
	runHooks(scriptName,"beforeShowMainDialog", null);

	//------------------------------
	// Display the dialog box.
	var go = dlgw.show();
	if ((go == 2) || (go == 4)) return(go);	// revert or load default settings

	// Get all fields
	sourcePathName = sourcePathGroup.sourcePathNameField.text;
	db_url = dbURLGroup.db_urlField.text;
	db_port = dbURLGroup.db_portField.text;
	db_name = dbNameGroup.db_nameField.text;
	db_tablename = dbTableGroup.db_tablenameField.text;
	db_user = dbNameGroup.db_userField.text;
	db_pw = dbTableGroup.db_pwField.text;
	domain = domainGroup.domainField.text;
	root_datapath = domainGroup.root_datapathField.text;

	if (PDFtextExtractCheck.value == true) PDFtextExtract = true;
	else PDFtextExtract = false;

	if (move2WEBdoneCheck.value == true) move2WEBdone = true;
	else move2WEBdone = false;

	// should current settings be saved?
	if ((go == 3) || (saveSettingsCheck.value == true)) {
		saveSettings = true;
		saveSettingsToFile();
	}
	else saveSettings = false;

	try { dlgw.close();} catch(e) {}

	// after show main dialog: run hook
	runHooks(scriptName,"afterShowMainDialog", {"go":go});

	return(go);
}


function checkDialogFields() {
	mainDialogErrorMessage = "";
	mainDialogErrorCode = 0;
	if ( db_url == "" ) {
		mainDialogErrorMessage = "'Database URL' may not be empty";
		mainDialogErrorCode = 1;
	}
	if ( db_port == "" ) {
		db_port = "3306";
	}
	if ( db_name == "" ) {
		mainDialogErrorMessage = "'Database Name' may not be empty";
		mainDialogErrorCode = 2;
	}
	if ( db_tablename == "" ) {
		mainDialogErrorMessage = "'Database Table Name' may not be empty";
		mainDialogErrorCode = 3;
	}
	if ( db_user == "" ) {
		mainDialogErrorMessage = "'User Name' may not be empty";
		mainDialogErrorCode = 4;
	}


	build_export_paths();	// enhance export paths with subpath

	return(mainDialogErrorCode);
}




function settingsFilePathWritable() {	// is settings path writable? otherwise return alternate settings path
	var fldr = new Folder(settingsFilePath);	// make sure the settings path exists
	var created = fldr.create();
	if (created == true) {	// we can write
		//alert("settings path: " + settingsFilePath);
		return(settingsFilePath);
	}
	else {
		// check if we can write into InDesign's scripting folder
		var checkFolder = new File(getScriptsPath() + "/" + settingsFilePathAlt);
		var checkFile = new File(getScriptsPath() + "/" + settingsFilePathAlt + "/writecheck");
		var created = checkFile.open("w");
		if (created == true) {	// we can write
			var settingsPath = checkFile.fsName;
			checkFile.close();
			//alert("writable: " + settingsPath);
			checkFile.remove();
			return(checkFolder.fsName);
		}
	}
	return(settingsFilePath);	// uuuups
}

function saveSettingsToFile() {
	// write the current settings to disk
	var settingsFile = new File(settingsFilePath + "/" + settingsFileName);
	var err = settingsFile.open("w");

	err = settingsFile.write("mainExportFolder=" + getURIName(mainExportFolder) + "\n");
	err = settingsFile.write("sourcePathName=" + getURIName(sourcePathName) + "\n");
	err = settingsFile.write("xslPathName=" + xslPathName + "\n");
	err = settingsFile.write("db_url=" + db_url + "\n");
	err = settingsFile.write("db_port=" + db_port + "\n");
	err = settingsFile.write("db_name=" + db_name + "\n");
	err = settingsFile.write("db_tablename=" + db_tablename + "\n");

	err = settingsFile.write("db_user=" + db_user + "\n");
	err = settingsFile.write("db_pw=" + db_pw + "\n");

	err = settingsFile.write("domain=" + domain + "\n");
	err = settingsFile.write("root_datapath=" + root_datapath + "\n");

	err = settingsFile.write("PDFtextExtract=" + PDFtextExtract + "\n");

	err = settingsFile.write("move2WEBdone=" + move2WEBdone + "\n");

	err = settingsFile.write("saveSettings=" + saveSettings + "\n");

	err = settingsFile.write("\n");	// ending empty line!!
	err = settingsFile.close();

}

function getURIName(path) {
	if ((path == null) || (path == "")) return("");
	try {
		var f = new File(path);
		return(f.fullName);
	} catch(e) {}
	return(path);
}


function loadSettingsFromFile() {
	// load settings from file
	var settingsFile = new File(settingsFilePath + "/" + settingsFileName);
	do {
		if (settingsFile.exists == false) break;
		var err = settingsFile.open("r");
	
		var line = settingsFile.readln();
		while (!settingsFile.eof) {
			if ((line != null) && (line != "") && (line.indexOf("#") != 0)) {
				var key = line.substr(0,line.indexOf("="));
				var arg = line.substr(line.indexOf("=")+1);
				var args_arr = new Array(key,arg);
				switch(args_arr[0]) {
					case "mainExportFolder": mainExportFolder = args_arr[1]; break;
					case "sourcePathName": sourcePathName = args_arr[1]; break;
					case "xslPathName": xslPathName = args_arr[1]; break;
					case "db_url": db_url = args_arr[1]; break;
					case "db_port": db_port = args_arr[1]; break;
					case "db_name": db_name = args_arr[1]; break;
					case "db_tablename": db_tablename = args_arr[1]; break;
					case "db_user": db_user = args_arr[1]; break;
					case "db_pw": db_pw = args_arr[1]; break;
					case "domain": domain = args_arr[1]; break;
					case "root_datapath": root_datapath = args_arr[1]; break;

					case "PDFtextExtract": if (args_arr[1] == "true") PDFtextExtract = true; else PDFtextExtract = false; break;

					case "move2WEBdone": if (args_arr[1] == "true") move2WEBdone = true; else move2WEBdone = false; break;
					case "saveSettings": if (args_arr[1] == "true") saveSettings = true; else saveSettings = false; break;
				}
			}
			line = settingsFile.readln();
		}
		err = settingsFile.close();
	} while (false);

	if ( mainExportFolder == "") {	// get from main settings.set
		loadMainSettingsFromFile();
		if ( mainExportFolder == "") mainExportFolder = "~/Export";
	}

	if (sourcePathName == "") sourcePathName = mainExportFolder + "/out/";
	var outfldr = new Folder(sourcePathName);
	sourcePathName = outfldr.fsName;

}


function loadMainSettingsFromFile() {
	// write the current settings to disk
	var settingsFile = new File(settingsFilePath + "/settings.set");
	if (settingsFile.exists == false) return;
	var err = settingsFile.open("r");

	var line = settingsFile.readln();
	while (!settingsFile.eof) {
		if ((line != null) && (line != "") && (line.indexOf("#") != 0)) {
			var key = line.substr(0,line.indexOf("="));
			var arg = line.substr(line.indexOf("=")+1);
			var args_arr = new Array(key,arg);
			switch(args_arr[0]) {
				case "mainExportFolder": mainExportFolder = args_arr[1]; break;
			}
		}
		line = settingsFile.readln();
	}
	err = settingsFile.close();
}


function build_export_paths() {
	// build new export paths
	// if everything is fine then create the full export paths
}


function getScriptsPath() {
	var fullpath = app.activeScript,
		scriptFile = new File(fullpath);
	return(scriptFile.path);
}
function getScriptName() {
	var fullpath = app.activeScript,
		scriptFile = new File(fullpath);
	return(scriptFile.name);
}


var messageWindow = null;
function messageClose() {
	if (!is_CS3_or_later) return;
	if (messageWindow != null) messageWindow.close();
}
function message(mess) {
	if (!is_CS3_or_later) return;
	if (messageWindow == null) messageCreate(mess);
	else messageWindow.messtext.text = mess;
	messageWindow.show();
}
function messageCreate(mess) {
	messageWindow = new Window('window', 'Progress', undefined, {maximizeButton:false, minimizeButton:false});
	messageWindow.messageGroup = messageWindow.add('group');
	messageWindow.messageGroup.orientation = 'row';
	messageWindow.messageGroup.margins = [50,30,50,30];
	messageWindow.messtext = messageWindow.messageGroup.add('statictext', undefined, mess);
}


function listProperties(the_prop_to_query) {
		var props_arr = the_prop_to_query;
		var props = the_prop_to_query.reflect.properties; 
		for (var i = 0; i < props.length; i++) { 
			alert("this property '" + props[i].name + "' is '" + props_arr[props[i].name] + "'"); 
		}
}

function startsWith (str,subs) { return (str.match("^"+subs)==subs); }

function  endsWith (str,subs) { return (str.match(subs+"$")==subs); }
